#!/usr/bin/env python3
import rclpy
from rclpy.node import Node
from rclpy.duration import Duration
from dbw_ford_msgs.msg import MiscCmd, TurnSignal, ParkingBrakeCmd

class BlinkerNode(Node):
    def __init__(self):
        super().__init__('mkz_blinker')
        self.topic = self.declare_parameter('topic', '/vehicle/misc_cmd').value
        self.mode  = self.declare_parameter('mode',  'right').value   # left|right|hazard|off
        self.duration_sec = float(self.declare_parameter('duration_sec', 3.0).value)
        self.rate_hz = float(self.declare_parameter('rate_hz', 20.0).value)

        self.pub = self.create_publisher(MiscCmd, self.topic, 10)
        mapping = {'off':('NONE',0), 'left':('LEFT',1), 'right':('RIGHT',2), 'hazard':('HAZARD',3)}
        enum_name, fallback = mapping.get(self.mode, ('RIGHT',2))
        enum_value = getattr(TurnSignal, enum_name, fallback)

        def mk_msg(val):
            msg = MiscCmd()
            msg.cmd.value = int(val)                     # TurnSignal value
            msg.pbrk.cmd = getattr(ParkingBrakeCmd,'NONE',0)  # leave parking brake unchanged
            return msg

        self.msg_on  = mk_msg(enum_value)
        self.msg_off = mk_msg(getattr(TurnSignal, 'NONE', 0))

        self.stop_time = self.get_clock().now() + Duration(seconds=self.duration_sec)
        self.timer = self.create_timer(1.0/self.rate_hz, self._tick)
        self.get_logger().info(f'Publishing {enum_name} on {self.topic} '
                               f'for {self.duration_sec:.1f}s @ {self.rate_hz:.0f} Hz')

    def _tick(self):
        if self.get_clock().now() < self.stop_time:
            self.pub.publish(self.msg_on)
        else:
            self.pub.publish(self.msg_off)
            rclpy.shutdown()

def main():
    rclpy.init()
    node = BlinkerNode()
    rclpy.spin(node)
