# FILE: ouster_alarm/README.md
# ouster_alarm


Trigger the right turn signal (or hazard) for a fixed duration when *motion* is detected within a 5 ft radius of the Ouster LiDAR.


> **Assumptions**: The vehicle is stationary. If the car is moving, you should compensate with odometry/IMU and/or a proper tracker.


## How it works
- Subscribes to a point cloud (`sensor_msgs/PointCloud2`, default `/ouster/points`).
- Crops a cylindrical ROI: radius = `5 ft` (configurable), z in `[-1.0, 1.5] m` (configurable).
- Voxelizes points into 10 cm cells and compares current vs previous frame occupancy.
- If the fraction of *new* voxels exceeds a threshold (default `0.30`) and there are enough voxels (default `50`), we consider that **motion**.
- On motion, repeatedly publishes `ds_dbw_msgs/TurnSignalCmd` at 10 Hz for `hold_seconds` (default `10s`).
- After hold, it publishes `NONE` and enforces a short cooldown.


## Build & Run
```bash
# In your ROS 2 workspace
cd ~/ros2_ws/src
# drop this folder here as ouster_alarm/


# Install dependencies
rosdep update
rosdep install --from-paths . --ignore-src -r -y


# Build
cd ..
colcon build --symlink-install --packages-select ouster_alarm
source install/setup.bash


# Run (node)
ros2 run ouster_alarm alarm_node


# Or via launch with overrides
ros2 launch ouster_alarm alarm.launch.py points_topic:=/ouster/points turn_signal_mode:=RIGHT
```


## Parameters
- `points_topic` (string, default `/ouster/points`): Ouster point cloud topic.
- `turn_signal_topic` (string, default `/vehicle/turn_signal_cmd`): Dataspeed turn signal command topic.
- `radius_feet` (double, default `5.0`): ROI radius around the sensor.
- `z_min`, `z_max` (double): bounding box in z.
- `voxel_size_m` (double, default `0.10`): voxel size.
- `diff_fraction_threshold` (double, default `0.30`): sensitivity threshold.
- `min_voxels` (int, default `50`): minimum occupancy to consider a valid detection.
- `hold_seconds` (double, default `10.0`): how long to keep the signal on.
- `cooldown_seconds` (double, default `2.0`): block retrigger for a bit after hold.
- `pub_rate_hz` (double, default `10.0`): publication rate while active.
- `turn_signal_mode` (string, `RIGHT` or `HAZARD`): which signal to activate.


## Notes
- If your message definition uses a field named `cmd` instead of `turn_signal`, the node auto-detects and sets the right field.
- If your Ouster topic is different (e.g. `/points`, `/os_cloud_node/points`), override `points_topic`.
- If you *truly* want hazards instead of a right turn signal, set `turn_signal_mode:=HAZARD`.
- For better motion discrimination or a moving vehicle, consider upgrading to a tracker (e.g. clustering + centroid tracking) and fusing IMU/odometry.


## Sanity checks
- Echo your Ouster topic to confirm fields:
```bash
ros2 topic echo /ouster/points | head -n 20
```
- Dry run the command publisher separately to confirm the car responds:
```bash
# RIGHT for 10s (manual)
ros2 topic pub -r 10 /vehicle/turn_signal_cmd ds_dbw_msgs/TurnSignalCmd '{turn_signal: 2}'
```
(If your message uses `cmd` instead of `turn_signal`, change the field accordingly.)


## License
MIT